#include "pch.h"

#include "Adrenalin.h"
#include "Common.hpp"
#include "Misc.hpp"
#include "Temp.hpp"
#include "CfgParse.hpp"
#include "LinksManager.h"
#include "Areas.h"

#include "ListOp.h"

void CArea::getNameDesc(std::string& nameDesc) const {
	const int	LINE_LEN   = 78;
	const int	PREFIX_LEN = 40;

	nameDesc = "   ";
	nameDesc.append( name );
    if (description.size() > 0) {
		nameDesc.append( 1, ' ' );
		if (nameDesc.size() < PREFIX_LEN-1) {
			nameDesc.append( PREFIX_LEN-1 - nameDesc.size(), '.' );
		}
		if (nameDesc.size() < PREFIX_LEN) {
			nameDesc.append( 1, ' ' );
		}
		std::string	alignedDesc;
		int	firstPrefixLen = nameDesc.size() > PREFIX_LEN ? nameDesc.size() : PREFIX_LEN;
		CListOp::alignDescription( description.c_str(), alignedDesc, LINE_LEN, "\r", PREFIX_LEN, firstPrefixLen );
		nameDesc.append( alignedDesc );
	}
}


bool CArea::isValidName(const tstring& areaName) {
	LPCTSTR	szName = areaName.c_str();
	while (*szName != _T('\0')) {
		if (*szName == _T('\\') || *szName == _T('/') || *szName == _T(':') ||
			*szName == _T('*') || *szName == _T('?') ||
			*szName == _T('<') || *szName == _T('>') ||
			*szName == _T('"'))
			return (false);
		szName++;
	}
	return (true);
}


void CArea::clearDescFile(LPCTSTR szFileName) const {
	ifstream     in;
	ofstream     tempout;
	TCHAR        szDescFileName[MAX_PATH];
	TCHAR        szTempFileName[MAX_PATH];
	TCHAR        szWorkFileName[MAX_PATH];
	TCHAR        szBuf[LINE_SIZE];
	char		szOemFileName[MAX_PATH];
	char		szOemBuf[MAX_PATH];
	int          Len;
	int          i;
	bool         fSkipLongDesc;
    
	// set file names
	lstrcpy( szDescFileName, workingPath.c_str() );
	lstrcat( szDescFileName, g_vars.m_szFilelistFileName );
	lstrcpy( szTempFileName, workingPath.c_str() );
	lstrcat( szTempFileName, _T("adrenalin.tmp") );
	Len = lstrlen( szFileName );

	// check name for spaces
	for (i = 0; szFileName[i] != _T('\0'); i++) {
		if (szFileName[i] == _T(' '))
			break;
	}
	if (szFileName[i] != _T('\0')) {
		lstrcpy( szWorkFileName, _T("\"") );
		lstrcat( szWorkFileName, szFileName );
		lstrcat( szWorkFileName, _T("\"") );
		Len += 2;
	}
	else
		lstrcpy( szWorkFileName, szFileName );
	// copy desc file into temp file, but skip lines with WorkFileName
	do {
		// check temp file name
		if (!GetTmpFileName( szTempFileName ))
			break;
		SetFileApisToOEM();
		CharToOem( szDescFileName, szOemFileName );
		in.open( szOemFileName );
		if (!in) {
			SetFileApisToANSI();
			break;
		}
		CharToOem( szTempFileName, szOemFileName );
		tempout.open( szOemFileName );
		if (!tempout) {
			in.close();
			SetFileApisToANSI();
			break;
		}
		CharToOem( szWorkFileName, szOemFileName );
		fSkipLongDesc = false;
		while (!in.eof()) {
			in.getline( szOemBuf, LINE_SIZE );
			if (!in)
				continue;
			if (strnicmp( szBuf, szOemFileName, Len) == 0)
				fSkipLongDesc = true;
			else {
				if (fSkipLongDesc) {
					if (*szOemBuf != ' ' && *szOemBuf != '>' && 
						*szOemBuf != '+') {
						fSkipLongDesc = false;
						tempout << szOemBuf << endl;
					}
				} else
					tempout << szOemBuf << endl;
			}
		}
		in.close();
		tempout.close();
		_tunlink( szDescFileName );
		_trename( szTempFileName, szDescFileName );
		break;
	} while (true);
}


CArea::CArea():
	group(NULL), path(), workingPath(), flags(),
	sendPriority(-1),
	receivePriority(-1),
	changed(false),
	fileFlags(),
	newFiles(),
	traffic(0),
	outboundDefined(false),
	outboundType(OT_BINKLEY)
{
}


CArea::~CArea() {
}


void CArea::setPath(const tstring& newPath) {
	path = newPath;

	// adjust working path

	if (path.size() > 0) {
		if (path[0] == TEXT('\\')) {
			workingPath = tstring(CAdrenalin::getStartupDrive()) + path;
		} else {
			if (path.size() < 2 || path[1] != TEXT(':')) {
				workingPath = tstring(CAdrenalin::getStartupPath()) + path;
			} else {
				workingPath = path;
			}
		}
	} else {
		workingPath = tstring(CAdrenalin::getStartupPath()) + path;
	}

	if (workingPath.size() > 0) {
		if (workingPath[workingPath.size()-1] != TEXT('\\')) {
			workingPath += TEXT("\\");
		}
	}
}


const tstring& CArea::getPath() const {
	return path;
}


void CArea::setFlag(int flag) {
	if (!hasFlag(flag)) {
		flags.push_back( flag );
	}
}


void CArea::resetFlag(int flag) {
	vector<int>::iterator	it = flags.begin();
	while (it != flags.end()) {
		if (flag == (*it)) {
			flags.erase( it );
			return;
		}
		it++;
	}
}


bool CArea::hasFlag(int flag) const {
	vector<int>::const_iterator	it = flags.begin();
	while (it != flags.end()) {
		if (flag == (*it))
			return true;
		it++;
	}
	return false;
}


bool CArea::canSend(ConstLinkPtr link) const {
	ConstTiePtr	tie = getTie(link);

	if (tie == NULL) {

		// not linked

		if (!link->canAccessGroup( group->getName() )) {
			// link cannot access area group
			return false;
		}
		if (link->getSendPriority() < sendPriority) {
			// link hasn't enough priority
			return (false);
		}
	} else {

		// linked

		if (tie->getRouteType() == ROUTE_NONE) {

			// no special routing

			if (link->getSendPriority() < sendPriority) {
				// link hasn't enough priority
				return (false);
			}
		} else {

			if ((tie->getRouteType() & ROUTE_SEND) == 0) {
				// routing doesn't allow link to send files
				return false;
			}

		}
	}

	return (true);
}


bool CArea::canReceive(ConstLinkPtr link) const {
	ConstTiePtr	tie = getTie(link);

	if (tie == NULL) {

		// not linked

		if (!link->canAccessGroup( group->getName() )) {
			// link cannot access area group
			return false;
		}
		if (link->getReceivePriority() < receivePriority) {
			// link hasn't enough priority
			return (false);
		}
	} else {

		// linked

		if (tie->getRouteType() == ROUTE_NONE) {

			// no special routing

			if (link->getReceivePriority() < receivePriority) {
				// link hasn't enough priority
				return (false);
			}
		} else {

			if ((tie->getRouteType() & ROUTE_RECEIVE) == 0) {
				// routing doesn't allow link to receive files
				return false;
			}
		}
	}

	return (true);
}


TiePtr CArea::getTie(ConstLinkPtr link) {
	int	count = ties.size();
	for (int i = 0; i < count; i++) {
		if (ties[i]->getLink() == link) {
			return ties[i];
		}
	}
	return NULL;
}


ConstTiePtr CArea::getTie(ConstLinkPtr link) const {
	int	count = ties.size();
	for (int i = 0; i < count; i++) {
		if (ties[i]->getLink() == link) {
			return ties[i];
		}
	}
	return NULL;
}


bool CArea::removeTie(ConstLinkPtr link) {
	for (VTies::iterator it = ties.begin(); it != ties.end(); ++it) {
		TiePtr	tie = (*it);
		if (tie->getLink() == link) {
			ties.erase( it );
			delete tie;
			return true;
		}
	}
	return false;
}


void CArea::clearDescFile(const VTString& vsFileNames) const {
	ifstream    in;
	ofstream    tempout;
	TCHAR       szDescFileName[MAX_PATH];
	TCHAR       szTempFileName[MAX_PATH];
	TCHAR       szBuf[LINE_SIZE];
	char		szOemFileName[MAX_PATH];
	char		szOemBuf[MAX_PATH];
	bool        fSkipLongDesc;
    
	// set file names
	lstrcpy( szDescFileName, workingPath.c_str() );
	lstrcat( szDescFileName, g_vars.m_szFilelistFileName );
	lstrcpy( szTempFileName, workingPath.c_str() );
	lstrcat( szTempFileName, TEXT("adrenalin.tmp") );

	// enclose names with spaces in quotes

	VTString					vsWorkFileNames;
	VTString::const_iterator	itName;

	for (itName = vsFileNames.begin(); itName != vsFileNames.end(); itName++) {
		if (HasSpaces((*itName).c_str())) {
			vsWorkFileNames.push_back( tstring(_T("\"")) + (*itName) + TEXT("\"") );
		}
		else {
			vsWorkFileNames.push_back( *itName );
		}
	}

	// copy desc file into temp file, but skip lines with WorkFileName
	do {
		// check temp file name
		if (!GetTmpFileName( szTempFileName ))
			break;
		SetFileApisToOEM();
		CharToOem( szDescFileName, szOemFileName );
		in.open( szOemFileName );
		if (!in) {
			SetFileApisToANSI();
			break;
		}
		CharToOem( szTempFileName, szOemFileName );
		tempout.open( szOemFileName );
		if (!tempout) {
			in.close();
			SetFileApisToANSI();
			break;
		}
		// prepare buffer for new line
		fSkipLongDesc = false;
		while (!in.eof()) {
			in.getline( szOemBuf, LINE_SIZE );
			if (!in)
				continue;
			OemToChar( szOemBuf, szBuf );
			// find name
			for (itName = vsWorkFileNames.begin(); itName != vsWorkFileNames.end(); itName++) {
				if (_tcsnicmp( szBuf, (*itName).c_str(), (*itName).length() ) == 0)
					break;
			}
			if (itName != vsWorkFileNames.end())
				fSkipLongDesc = true;
			else {
				if (fSkipLongDesc) {
					if (*szOemBuf != ' ' && *szOemBuf != '>' && 
						*szOemBuf != '+') {
						fSkipLongDesc = false;
						tempout << szOemBuf << endl;
					}
				} else
					tempout << szOemBuf << endl;
			}
		}
		in.close();
		tempout.close();
		_tunlink( szDescFileName );
		_trename( szTempFileName, szDescFileName );
		break;
	} while (true);
}


void CArea::updateDescFile(LPCTSTR szFileName, LPCTSTR szFileDesc, filesize_t fileSize,
                           const VTString& listLDesc) const
{
	ofstream	out;
	ifstream    in;
	ofstream    tempout;
	TCHAR       szDescFileName[MAX_PATH];
	TCHAR       szTempFileName[MAX_PATH];
	TCHAR       szWorkFileName[MAX_PATH];
	TCHAR       szBuf[LINE_SIZE];
	char		szOemBuf[LINE_SIZE];
	char		szOemFileName[MAX_PATH];
	int         Len;
	int         i;
	bool        fSkipLongDesc;
    
	// set file names
	lstrcpy( szDescFileName, workingPath.c_str() );
	lstrcat( szDescFileName, g_vars.m_szFilelistFileName );
	lstrcpy( szTempFileName, workingPath.c_str() );
	lstrcat( szTempFileName, TEXT("adrenalin.tmp") );
	Len = lstrlen( szFileName );

	// check name for spaces
	for (i = 0; szFileName[i] != _T('\0'); i++) {
		if (szFileName[i] == _T(' '))
			break;
	}
	if (szFileName[i] != _T('\0')) {
		lstrcpy( szWorkFileName, _T("\"") );
		lstrcat( szWorkFileName, szFileName );
		lstrcat( szWorkFileName, _T("\"") );
		Len += 2;
	}
	else
		lstrcpy( szWorkFileName, szFileName );

	// copy desc file into temp file, but skip lines with WorkFileName
	BOOL fSuccessOverwrite = TRUE;

	do {
		// check temp file name
		if (!GetTmpFileName( szTempFileName ))
			break;
		CharToOem( szDescFileName, szOemFileName );
		SetFileApisToOEM();
		in.open( szOemFileName );
		if (!in) {
			SetFileApisToANSI();
			break;
		}
		CharToOem( szTempFileName, szOemFileName );
		tempout.open( szOemFileName );
		if (!tempout) {
			in.close();
			SetFileApisToANSI();
			break;
		}
		SetFileApisToANSI();
		CharToOem( szWorkFileName, szOemFileName );
		fSkipLongDesc = false;
		while (!in.eof()) {
			in.getline( szOemBuf, LINE_SIZE );
			if (!in) {
				if (!in.eof())
					break;
				continue;
			}

			// skip empty lines
			if (*szOemBuf == '\0')
				continue;

			if ((strnicmp( szOemBuf, szOemFileName, Len) == 0) && ((szOemBuf[Len] == ' ') || (szOemBuf[Len] == '\0')))
				fSkipLongDesc = true;
			else {
				if (fSkipLongDesc) {
					if (*szOemBuf != ' ' && *szOemBuf != '>' && *szOemBuf != '+') {
						fSkipLongDesc = false;
						tempout << szOemBuf << endl;
						if (!tempout) {
							fSuccessOverwrite = false;
							break;
						}
					}
				} else {
					tempout << szOemBuf << endl;
					if (!tempout) {
						fSuccessOverwrite = false;
						break;
					}
				}
			}
		}
		in.close();
		tempout.close();
		if (fSuccessOverwrite) {
			_tunlink( szDescFileName );
			_trename( szTempFileName, szDescFileName );
		} else {
			_tunlink( szTempFileName );
			g_vars.m_log << _T("! error updating ") << szDescFileName << _T("\r\n");
		}
		break;
	} while (true);
    
	CharToOem( szDescFileName, szOemFileName );
	out.open( szOemFileName, ios::app );
	if (!out)
	        out.open( szOemFileName );
	if (!out)
		return;
	CharToOem( szWorkFileName, szOemFileName );

	CharToOem( g_vars.m_fileListFormat.apply( szFileName, szFileDesc, fileSize).c_str(), szOemBuf );

	/*
	out << szOemFileName;
	if (*szFileDesc != _T('\0')) {
		while (Len < 12) {
			out << " ";
			Len++;
		}
		out << " ";
		CharToOem( szFileDesc, szOemBuf );
		out << szOemBuf;
	}
	*/
	out << szOemBuf;
	out << endl;


	VTString::const_iterator	it;
	for (it = listLDesc.begin(); it != listLDesc.end(); it++) {
		ConcatStrings( szBuf, g_vars.m_szLongDescPrefix, (*it).c_str() );
		CharToOem( szBuf, szOemBuf );
		out << szOemBuf << endl;
	}
	out.close();
}
